@extends('layouts.app')

@section('title', 'Store')

@section('content')
<div class="store-container">
    <div class="container">
        <!-- Store Header -->
        <div class="store-header">
            <h1 class="store-title">Game Store</h1>
            <p class="store-subtitle">Enhance your gaming experience with premium packages and exclusive content</p>
        </div>

        <!-- Controls Section -->
        <div class="store-controls">
            <div class="controls-left">
                @if($servers->count() > 1)
                <div class="server-selector">
                    <label for="serverSelect">
                        <i class="fas fa-server"></i>
                        Server
                    </label>
                    <select id="serverSelect" class="server-select">
                        <option value="">All Servers</option>
                        @foreach($servers as $serverOption)
                            <option value="{{ $serverOption->id }}" 
                                    {{ $server && $server->id == $serverOption->id ? 'selected' : '' }}>
                                {{ $serverOption->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                @endif
            </div>

            <div class="controls-center">
                <div class="search-bar">
                    <i class="fas fa-search search-icon"></i>
                    <input type="text" id="packageSearch" placeholder="Search packages..." class="search-input">
                </div>
            </div>

            <div class="controls-right">
                <div class="category-filter">
                    <div class="filter-tabs">
                        <button class="filter-tab active" data-category="all">
                            <span>All</span>
                            <span class="count">{{ $packages->count() }}</span>
                        </button>
                        @foreach($categories as $category)
                        <button class="filter-tab" data-category="{{ $category }}">
                            <span>{{ ucfirst($category) }}</span>
                            <span class="count">{{ $packages->where('type', $category)->count() }}</span>
                        </button>
                        @endforeach
                    </div>
                </div>
            </div>
        </div>

        <!-- Packages Grid -->
        <div class="packages-grid" id="packagesGrid">
            @foreach($packages as $package)
            <div class="package-card {{ $package->type }} {{ Str::contains(strtolower($package->name), 'elite') ? 'elite-' . $package->type : '' }}" data-category="{{ $package->type }}" data-name="{{ strtolower($package->name) }}">
                <div class="package-image" style="background-image: url('{{ asset('storage/' . $package->image_path ?? 'images/store/default.jpg') }}')">
                </div>
                
                <div class="package-category package-category-{{ $package->type }}">
                    {{ ucfirst($package->type) }}
                </div>
                
                @if(Str::contains(strtolower($package->name), 'elite'))
                    <div class="package-badge">
                        <i class="fas fa-crown"></i> Elite
                    </div>
                @endif
                
                <div class="package-content">
                    <div class="package-header">
                        <h3 class="package-title">{{ $package->name }}</h3>
                        <div class="package-price">{{ currency_symbol(config_value('store_currency', 'GBP')) }}{{ number_format($package->price, 2) }}</div>
                    </div>

                    <div class="package-features">
                        <ul>
                            @foreach(array_slice($package->features, 0, 4) as $feature)
                            <li>{{ $feature }}</li>
                            @endforeach
                        </ul>
                        
                        @if(count($package->features) > 4)
                        <div class="feature-more">
                            +{{ count($package->features) - 4 }} more features
                        </div>
                        @endif
                    </div>

                    <div class="package-actions">
                        <a href="{{ route('store.show', ['slug' => $package->slug, 'server_id' => $server ? $server->id : null]) }}" 
                           class="btn-view">
                            <span>View Details</span>
                            <i class="fas fa-arrow-right"></i>
                        </a>
                    </div>
                </div>
            </div>
            @endforeach
        </div>

        <!-- Empty State -->
        <div class="empty-state" id="emptyState" style="display: none;">
            <div class="empty-icon">
                <i class="fas fa-search"></i>
            </div>
            <h3>No packages found</h3>
            <p>Try adjusting your search or category filter to find what you're looking for.</p>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const serverSelect = document.getElementById('serverSelect');
    const categoryTabs = document.querySelectorAll('.filter-tab');
    const searchInput = document.getElementById('packageSearch');
    const packagesGrid = document.getElementById('packagesGrid');
    const packageCards = document.querySelectorAll('.package-card');
    const emptyState = document.getElementById('emptyState');
    
    let currentCategory = 'all';
    let currentSearch = '';

    // Server selection
    if (serverSelect) {
        serverSelect.addEventListener('change', function() {
            const selectedServerId = this.value;
            const url = new URL(window.location);
            
            if (selectedServerId) {
                url.searchParams.set('server_id', selectedServerId);
            } else {
                url.searchParams.delete('server_id');
            }
            
            window.location.href = url.toString();
        });
    }

    // Category filtering
    categoryTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            // Update active tab
            categoryTabs.forEach(t => t.classList.remove('active'));
            this.classList.add('active');
            
            currentCategory = this.dataset.category;
            filterPackages();
        });
    });

    // Search functionality
    searchInput.addEventListener('input', function() {
        currentSearch = this.value.toLowerCase();
        filterPackages();
    });

    function filterPackages() {
        let visibleCount = 0;

        packageCards.forEach(card => {
            const cardCategory = card.dataset.category;
            const cardName = card.dataset.name;
            
            const categoryMatch = currentCategory === 'all' || cardCategory === currentCategory;
            const searchMatch = currentSearch === '' || cardName.includes(currentSearch);
            
            if (categoryMatch && searchMatch) {
                card.style.display = 'flex';
                visibleCount++;
            } else {
                card.style.display = 'none';
            }
        });

        // Show/hide empty state
        if (visibleCount === 0) {
            emptyState.style.display = 'flex';
            packagesGrid.style.display = 'none';
        } else {
            emptyState.style.display = 'none';
            packagesGrid.style.display = 'grid';
        }
    }
});
</script>
@endpush
@endsection